<?php
/**
 * Admin Authentication Helper
 * Handles secure admin authentication with password hashing
 */

class AdminAuth {
    private $pdo;
    private $maxLoginAttempts = 5;
    private $lockoutDuration = 900; // 15 minutes in seconds
    
    public function __construct($pdo) {
        $this->pdo = $pdo;
    }
    
    /**
     * Verify admin login
     */
    public function login($username, $password) {
        try {
            // Check if account is locked
            $stmt = $this->pdo->prepare("
                SELECT id, username, password_hash, locked_until, login_attempts 
                FROM binance_p2p_admin_users 
                WHERE username = ?
            ");
            $stmt->execute([$username]);
            $user = $stmt->fetch(PDO::FETCH_ASSOC);
            
            if (!$user) {
                // User doesn't exist - don't reveal this
                $this->recordFailedAttempt($username);
                return ['success' => false, 'error' => 'Invalid credentials'];
            }
            
            // Check if account is locked
            if ($user['locked_until'] && strtotime($user['locked_until']) > time()) {
                $remaining = strtotime($user['locked_until']) - time();
                return [
                    'success' => false, 
                    'error' => "Account locked. Try again in " . ceil($remaining / 60) . " minutes."
                ];
            }
            
            // Verify password
            if (!password_verify($password, $user['password_hash'])) {
                $this->recordFailedAttempt($username);
                return ['success' => false, 'error' => 'Invalid credentials'];
            }
            
            // Successful login - reset attempts and update last login
            $this->resetLoginAttempts($username);
            $this->updateLastLogin($user['id']);
            
            return [
                'success' => true,
                'user_id' => $user['id'],
                'username' => $user['username']
            ];
            
        } catch (PDOException $e) {
            error_log("Admin login error: " . $e->getMessage());
            return ['success' => false, 'error' => 'Login failed. Please try again.'];
        }
    }
    
    /**
     * Record failed login attempt
     */
    private function recordFailedAttempt($username) {
        try {
            $stmt = $this->pdo->prepare("
                UPDATE binance_p2p_admin_users 
                SET login_attempts = login_attempts + 1,
                    locked_until = CASE 
                        WHEN login_attempts + 1 >= ? THEN DATE_ADD(NOW(), INTERVAL ? SECOND)
                        ELSE locked_until
                    END
                WHERE username = ?
            ");
            $stmt->execute([$this->maxLoginAttempts, $this->lockoutDuration, $username]);
        } catch (PDOException $e) {
            error_log("Error recording failed login attempt: " . $e->getMessage());
        }
    }
    
    /**
     * Reset login attempts after successful login
     */
    private function resetLoginAttempts($username) {
        try {
            $stmt = $this->pdo->prepare("
                UPDATE binance_p2p_admin_users 
                SET login_attempts = 0, locked_until = NULL 
                WHERE username = ?
            ");
            $stmt->execute([$username]);
        } catch (PDOException $e) {
            error_log("Error resetting login attempts: " . $e->getMessage());
        }
    }
    
    /**
     * Update last login timestamp and IP
     */
    private function updateLastLogin($userId) {
        try {
            $ip = $_SERVER['REMOTE_ADDR'] ?? 'unknown';
            $stmt = $this->pdo->prepare("
                UPDATE binance_p2p_admin_users 
                SET last_login_at = NOW(), last_login_ip = ? 
                WHERE id = ?
            ");
            $stmt->execute([$ip, $userId]);
        } catch (PDOException $e) {
            error_log("Error updating last login: " . $e->getMessage());
        }
    }
    
    /**
     * Check if user is logged in
     */
    public function isLoggedIn() {
        if (!isset($_SESSION['admin_logged_in']) || !$_SESSION['admin_logged_in']) {
            return false;
        }
        
        // Check session timeout (30 minutes)
        if (isset($_SESSION['admin_login_time']) && 
            (time() - $_SESSION['admin_login_time']) > 1800) {
            $this->logout();
            return false;
        }
        
        return true;
    }
    
    /**
     * Require login (redirect if not logged in)
     */
    public function requireLogin() {
        if (!$this->isLoggedIn()) {
            header('Location: login.php');
            exit();
        }
    }
    
    /**
     * Logout
     */
    public function logout() {
        $_SESSION['admin_logged_in'] = false;
        $_SESSION['admin_user_id'] = null;
        $_SESSION['admin_username'] = null;
        $_SESSION['admin_login_time'] = null;
        session_destroy();
    }
    
    /**
     * Change password
     */
    public function changePassword($username, $oldPassword, $newPassword) {
        try {
            // Verify old password
            $loginResult = $this->login($username, $oldPassword);
            if (!$loginResult['success']) {
                return ['success' => false, 'error' => 'Current password is incorrect'];
            }
            
            // Validate new password
            if (strlen($newPassword) < 8) {
                return ['success' => false, 'error' => 'Password must be at least 8 characters'];
            }
            
            // Hash and update password
            $hash = password_hash($newPassword, PASSWORD_BCRYPT);
            $stmt = $this->pdo->prepare("
                UPDATE binance_p2p_admin_users 
                SET password_hash = ? 
                WHERE username = ?
            ");
            $stmt->execute([$hash, $username]);
            
            return ['success' => true];
            
        } catch (PDOException $e) {
            error_log("Error changing password: " . $e->getMessage());
            return ['success' => false, 'error' => 'Failed to change password'];
        }
    }
}

