<?php
/**
 * Log Viewer Script
 * View recent error logs and Bridge API logs
 * 
 * Usage: php api/view-logs.php [type] [lines]
 *   type: 'bridge', 'websocket', 'all' (default: 'all')
 *   lines: number of lines to show (default: 50)
 * 
 * Example:
 *   php api/view-logs.php bridge 100
 *   php api/view-logs.php websocket 50
 */

if (php_sapi_name() !== 'cli') {
    http_response_code(403);
    die('This script can only be run from command line.');
}

$baseDir = dirname(__DIR__);
$type = $argv[1] ?? 'all';
$lines = (int)($argv[2] ?? 50);

echo "=== Log Viewer ===\n";
echo "Type: $type\n";
echo "Lines: $lines\n\n";

// Common log file locations
$logFiles = [
    'php_error' => [
        '/var/log/php_errors.log',
        '/var/log/apache2/error.log',
        '/var/log/nginx/error.log',
        $baseDir . '/logs/error.log',
        $baseDir . '/api/error_log',
    ],
    'websocket' => [
        $baseDir . '/logs/websocket-chat.log',
    ],
];

// Find PHP error log
$phpErrorLog = null;
foreach ($logFiles['php_error'] as $logFile) {
    if (file_exists($logFile) && is_readable($logFile)) {
        $phpErrorLog = $logFile;
        break;
    }
}

// Find WebSocket log
$websocketLog = null;
foreach ($logFiles['websocket'] as $logFile) {
    if (file_exists($logFile) && is_readable($logFile)) {
        $websocketLog = $logFile;
        break;
    }
}

// Show Bridge API logs
if ($type === 'bridge' || $type === 'all') {
    echo "=== Bridge API Logs ===\n";
    if ($phpErrorLog) {
        echo "Reading from: $phpErrorLog\n";
        $command = "grep 'Bridge API' " . escapeshellarg($phpErrorLog) . " | tail -$lines";
        $output = shell_exec($command);
        if ($output) {
            echo $output;
        } else {
            echo "No Bridge API logs found in $phpErrorLog\n";
        }
    } else {
        echo "PHP error log not found. Checked locations:\n";
        foreach ($logFiles['php_error'] as $logFile) {
            echo "  - $logFile\n";
        }
        echo "\nTo find your PHP error log, run: php -i | grep error_log\n";
    }
    echo "\n";
}

// Show WebSocket logs
if ($type === 'websocket' || $type === 'all') {
    echo "=== WebSocket Chat Logs ===\n";
    if ($websocketLog) {
        echo "Reading from: $websocketLog\n";
        $command = "tail -$lines " . escapeshellarg($websocketLog);
        $output = shell_exec($command);
        if ($output) {
            echo $output;
        } else {
            echo "Log file is empty\n";
        }
    } else {
        echo "WebSocket log not found at: " . $baseDir . "/logs/websocket-chat.log\n";
    }
    echo "\n";
}

// Show all PHP errors
if ($type === 'all' && $phpErrorLog) {
    echo "=== Recent PHP Errors (Last $lines lines) ===\n";
    $command = "tail -$lines " . escapeshellarg($phpErrorLog);
    $output = shell_exec($command);
    if ($output) {
        echo $output;
    } else {
        echo "No recent errors\n";
    }
    echo "\n";
}

// Show PHP error log location
echo "=== PHP Error Log Location ===\n";
$phpInfo = shell_exec('php -i 2>/dev/null | grep "error_log" | head -1');
if ($phpInfo) {
    echo trim($phpInfo) . "\n";
} else {
    echo "Run: php -i | grep error_log\n";
}
echo "\n";

// Show log file sizes
echo "=== Log File Sizes ===\n";
if ($phpErrorLog) {
    $size = filesize($phpErrorLog);
    echo "PHP Error Log: " . number_format($size / 1024, 2) . " KB\n";
}
if ($websocketLog) {
    $size = filesize($websocketLog);
    echo "WebSocket Log: " . number_format($size / 1024, 2) . " KB\n";
}
echo "\n";

echo "=== Done ===\n";

