<?php
/**
 * Binance P2P Settings Helper
 * Manages settings for Binance P2P automation
 */

class BinanceP2PSettings {
    private $pdo;
    
    public function __construct($pdo) {
        $this->pdo = $pdo;
    }
    
    /**
     * Get setting value
     */
    public function get($key, $default = null) {
        try {
            $stmt = $this->pdo->prepare("SELECT setting_value FROM binance_p2p_settings WHERE setting_key = ?");
            $stmt->execute([$key]);
            $result = $stmt->fetch(PDO::FETCH_ASSOC);
            return $result ? $result['setting_value'] : $default;
        } catch (PDOException $e) {
            error_log("Error getting setting $key: " . $e->getMessage());
            return $default;
        }
    }
    
    /**
     * Set setting value
     */
    public function set($key, $value) {
        try {
            $stmt = $this->pdo->prepare("
                INSERT INTO binance_p2p_settings (setting_key, setting_value) 
                VALUES (?, ?)
                ON DUPLICATE KEY UPDATE setting_value = ?, updated_at = NOW()
            ");
            $stmt->execute([$key, $value, $value]);
            return true;
        } catch (PDOException $e) {
            error_log("Error setting $key: " . $e->getMessage());
            return false;
        }
    }
    
    /**
     * Get Bridge customer ID
     */
    public function getBridgeCustomerId() {
        return $this->get('bridge_customer_id');
    }
    
    /**
     * Set Bridge customer ID
     */
    public function setBridgeCustomerId($customerId) {
        return $this->set('bridge_customer_id', $customerId);
    }
    
    /**
     * Get Bridge external account ID
     */
    public function getBridgeExternalAccountId() {
        return $this->get('bridge_external_account_id');
    }
    
    /**
     * Set Bridge external account ID
     */
    public function setBridgeExternalAccountId($accountId) {
        return $this->set('bridge_external_account_id', $accountId);
    }
    
    /**
     * Check if auto payment is enabled
     */
    public function isAutoPaymentEnabled() {
        return $this->get('auto_payment_enabled', '1') === '1';
    }
    
    /**
     * Enable/disable auto payment
     */
    public function setAutoPaymentEnabled($enabled) {
        return $this->set('auto_payment_enabled', $enabled ? '1' : '0');
    }
    
    /**
     * Get low balance threshold
     */
    public function getLowBalanceThreshold() {
        return floatval($this->get('low_balance_threshold', '100'));
    }
    
    /**
     * Get Binance API credentials
     */
    public function getBinanceApiKey() {
        return $this->get('binance_api_key');
    }
    
    public function getBinanceSecretKey() {
        return $this->get('binance_secret_key');
    }
    
    /**
     * Set Binance API credentials
     */
    public function setBinanceCredentials($apiKey, $secretKey) {
        $this->set('binance_api_key', $apiKey);
        $this->set('binance_secret_key', $secretKey);
    }
    
    /**
     * Mark low balance as notified
     */
    public function markLowBalanceNotified($notified = true) {
        $this->set('low_balance_notified', $notified ? '1' : '0');
        if ($notified) {
            $this->set('last_balance_check', date('Y-m-d H:i:s'));
        }
    }
    
    /**
     * Check if low balance was already notified
     */
    public function isLowBalanceNotified() {
        return $this->get('low_balance_notified', '0') === '1';
    }
}

