<?php
/**
 * Debug Script for Binance API
 * Captures raw request and response data for Binance support
 */

// Set base directory
$baseDir = dirname(__DIR__);

// Load required files
require_once $baseDir . '/includes/config.php';
require_once $baseDir . '/includes/binance-api.php';
require_once $baseDir . '/includes/binance-settings.php';

// Get settings
$settings = new BinanceP2PSettings($pdo);

// Initialize Binance API
$binance = new BinanceP2PAPI(
    $settings->get('binance_api_key'),
    $settings->get('binance_secret_key')
);

// Enable detailed logging
error_reporting(E_ALL);
ini_set('display_errors', 1);

echo "=== Binance API Debug Script ===\n\n";
echo "This script will capture raw API requests and responses.\n";
echo "Copy the output below and send it to Binance Support.\n\n";
echo str_repeat("=", 80) . "\n\n";

// Test 1: Get User Order List
echo "TEST 1: GET /sapi/v1/c2c/orderMatch/listUserOrderHistory\n";
echo str_repeat("-", 80) . "\n";
$testOrderNo = '22829218531025494016'; // Use a recent order number

$response1 = $binance->getUserOrderList(1, 5);
echo "\n--- REQUEST DETAILS ---\n";
echo "Method: GET\n";
echo "Endpoint: /sapi/v1/c2c/orderMatch/listUserOrderHistory\n";
echo "Parameters: page=1, rows=5\n";
echo "API Key: " . substr($settings->get('binance_api_key'), 0, 10) . "...\n";
echo "\n--- RESPONSE ---\n";
echo "HTTP Code: " . ($response1['http_code'] ?? 'N/A') . "\n";
if (isset($response1['error'])) {
    echo "Error: " . $response1['error'] . "\n";
}
echo "Response Data:\n";
echo json_encode($response1, JSON_PRETTY_PRINT | JSON_UNESCAPED_UNICODE);
echo "\n\n";

// Test 2: Get User Order Detail (GET)
echo str_repeat("=", 80) . "\n";
echo "TEST 2: GET /sapi/v1/c2c/orderMatch/getUserOrderDetail\n";
echo str_repeat("-", 80) . "\n";
echo "Order No: $testOrderNo\n";

// Capture the raw request by modifying the method temporarily
$response2 = $binance->signedRequest('GET', '/sapi/v1/c2c/orderMatch/getUserOrderDetail', [
    'orderNo' => $testOrderNo
]);

echo "\n--- REQUEST DETAILS ---\n";
echo "Method: GET\n";
echo "Endpoint: /sapi/v1/c2c/orderMatch/getUserOrderDetail\n";
echo "Parameters: orderNo=$testOrderNo\n";
echo "Full URL: https://api.binance.com/sapi/v1/c2c/orderMatch/getUserOrderDetail?orderNo=$testOrderNo&timestamp=...&signature=...\n";
echo "\n--- RESPONSE ---\n";
echo "HTTP Code: " . ($response2['http_code'] ?? 'N/A') . "\n";
if (isset($response2['error'])) {
    echo "Error: " . $response2['error'] . "\n";
    echo "Error Code: " . ($response2['code'] ?? 'N/A') . "\n";
}
echo "Response Data:\n";
echo json_encode($response2, JSON_PRETTY_PRINT | JSON_UNESCAPED_UNICODE);
echo "\n\n";

// Test 3: Get User Order Detail (POST)
echo str_repeat("=", 80) . "\n";
echo "TEST 3: POST /sapi/v1/c2c/orderMatch/getUserOrderDetail\n";
echo str_repeat("-", 80) . "\n";
echo "Order No: $testOrderNo\n";

$response3 = $binance->signedRequest('POST', '/sapi/v1/c2c/orderMatch/getUserOrderDetail', [
    'orderNo' => $testOrderNo
]);

echo "\n--- REQUEST DETAILS ---\n";
echo "Method: POST\n";
echo "Endpoint: /sapi/v1/c2c/orderMatch/getUserOrderDetail\n";
echo "Body Parameters: orderNo=$testOrderNo\n";
echo "Full URL: https://api.binance.com/sapi/v1/c2c/orderMatch/getUserOrderDetail\n";
echo "\n--- RESPONSE ---\n";
echo "HTTP Code: " . ($response3['http_code'] ?? 'N/A') . "\n";
if (isset($response3['error'])) {
    echo "Error: " . $response3['error'] . "\n";
    echo "Error Code: " . ($response3['code'] ?? 'N/A') . "\n";
}
echo "Response Data:\n";
echo json_encode($response3, JSON_PRETTY_PRINT | JSON_UNESCAPED_UNICODE);
echo "\n\n";

// Test 4: Check order structure from listUserOrderHistory
echo str_repeat("=", 80) . "\n";
echo "TEST 4: Analyzing Order Structure from listUserOrderHistory\n";
echo str_repeat("-", 80) . "\n";

$orders = $response1['data'] ?? [];
if (!empty($orders) && isset($orders[0])) {
    $firstOrder = $orders[0];
    echo "First Order Keys: " . implode(', ', array_keys($firstOrder)) . "\n\n";
    echo "Order Number: " . ($firstOrder['orderNumber'] ?? 'N/A') . "\n";
    echo "Order Status: " . ($firstOrder['orderStatus'] ?? 'N/A') . "\n";
    echo "Payment Method Name: " . ($firstOrder['payMethodName'] ?? 'N/A') . "\n\n";
    
    echo "--- CHECKING FOR PAYMENT DETAILS ---\n";
    $paymentFields = ['paymentMethodFields', 'payAccount', 'buyerPaymentInfo', 'paymentDetails', 'paymentInfo', 'iban', 'IBAN'];
    $found = false;
    foreach ($paymentFields as $field) {
        if (isset($firstOrder[$field])) {
            echo "✓ Found: $field\n";
            echo "  Value: " . (is_array($firstOrder[$field]) ? json_encode($firstOrder[$field]) : $firstOrder[$field]) . "\n";
            $found = true;
        }
    }
    
    if (!$found) {
        echo "✗ No payment detail fields found in order response\n";
    }
    
    echo "\n--- FULL ORDER STRUCTURE ---\n";
    echo json_encode($firstOrder, JSON_PRETTY_PRINT | JSON_UNESCAPED_UNICODE);
}

echo "\n\n";
echo str_repeat("=", 80) . "\n";
echo "END OF DEBUG OUTPUT\n";
echo "Please copy everything above and send it to Binance Support\n";
echo str_repeat("=", 80) . "\n";

